--DROP PROCEDURE App.NVCC_usp_UsageData2

USE NWIC_PanelMgmt
GO

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE PROCEDURE App.NVCC_usp_UsageData2
	-- @timegroup is a character parameter which can take values NULL, 'day', 'week',
	-- 'month', 'quarter', 'year', or 'ever'. NULL is equivalent to 'ever'. Any other value
	-- will result in undefind behavior.
	@timegroup varchar(max) NULL,
	-- @statistic is a character parameter which can take values 'count' or 'cumulative'.
	-- Any other value will result in undefined behavior.
	-- For 'count', the value N represents number within that timegroup and any
	-- other grouping categories
	-- For 'cumulative', the value N represents the cumulative number, over time,
	-- up to that timepoint, separated by any other grouping categories
	@statistic varchar(max) NULL,
	-- @sta3n is a character parameter which can either NULL, a single station number,
	-- or a comma demlimited list of station numbers. NULL implies that the data for
	-- all stations should be returned/used and that no stratification by station should
	-- be done. A single station number (as a string) will return data for only that
	-- station. A comma delimited list of station numbers will restrict the returned data
	-- to those present in the list and any aggregation or ordination will be done
	-- separately for each station.
	@sta3n varchar(max) NULL,
	-- @userid is a character parameter which can either NULL, a single user identifier,
	-- or a comma demlimited list of user identifiers. A user identifier is a login,
	-- including the domain. NULL implies that the data for all userss stations should
	-- be returned/used and that no stratification by user should be done. A userid
	-- will return data for only that user. A comma delimited list of users will
	-- restrict the returned data to those present in the list and any aggregation
	-- or ordination will be done separately for each user.
	@userid varchar(max) NULL
AS
--==================================================================================
--Requestor             : Non Va Coordinated Care Project
--Author                : Brian Diggs
--Object/SP Name        : App.NVCC_usp_UsageData2
--Server                : DNS.URL        
--Data Base             : NWIC_PanelMgmt
--Schema                : app
--Report                : N/A
--Folder                : N/A
--Report Loc            : N/A
--Job                   : None
--Job Loc               : None
--Note                  : 
--Date Created          : 08-03-2016
--
--Last Changed          : 
--Last Changed By		: 
--Reason For Change		: 
--
--Purpose               : To get aggregated, subsetted production counts.
--==================================================================================
--Uses: NWIC_PanelMgmt.App.udf_SplitString
--Uses: NWIC_PanelMgmt.App.vNVCC_ProductionAccessLog
--Uses: NWIC_PanelMgmt.App.NVCC_Users
--Uses: CDWWork.Patient.Patient
--Uses: CDWWork.SStaff.SStaff

BEGIN
	-- First, create a temporary table, #LogData, which pulls records from
	-- App.NVCC_ProductionAccessLog.
	--
	-- The AccessDateTime and UserID are kept from the log, and the
	-- PatientSID is used to determine the Sta3n for that patient.
	--
	-- This list is further restricted to those stations and users specified
	-- by the parameters. This is accomplished by converting the parameter
	-- string to a table using App.udf_SplitString. Only the value is kept
	-- and, for the station, converted to a smallint to match the datatype
	-- elsewhere. An input of NULL will give a single row table with value
	-- NULL. These conversions are done as common table expressions.
	-- These table are INNER JOIN-ed to the data, with the clause that
	-- everything is kept if the value is NULL.
	--
	-- UserID and Sta3n are converted to NULL if their respective parameters
	-- are NULL. This indicates that there is no stratification done (or
	-- possible) on those variables. The TimeGroup column contains a date
	-- which is the start of the interval corresponding to the requested
	-- aggregation level (day, start of week, start of month, start of
	-- quarter, start of year).
	WITH
		Sta3n AS (SELECT CONVERT(smallint, Value) AS Sta3n FROM App.udf_SplitString(@sta3n, ',')),
		UserID AS (SELECT Value AS UserID FROM App.udf_SplitString(@userid, ','))
		SELECT
			IIF(@userid IS NULL, NULL, al.UserID) AS UserID,
			IIF(@sta3n IS NULL, NULL, p.sta3n) AS Sta3n,
			CASE @timegroup
				WHEN 'day' THEN
					CONVERT(DATE, al.AccessDateTime)
				WHEN 'week' THEN
					CONVERT(DATE, DATEADD(DAY, 7 * (DATEDIFF(DAY, '1990-01-07', al.AccessDateTime) / 7), '1990-01-07'))
				WHEN 'month' THEN
					DATEFROMPARTS(DATEPART(YEAR, al.AccessDateTime), DATEPART(MONTH, al.AccessDateTime), 1)
				WHEN 'quarter' THEN
					DATEFROMPARTS(DATEPART(YEAR, al.AccessDateTime), (((DATEPART(MONTH, al.AccessDateTime) - 1) / 3) * 3) + 1, 1)
				WHEN 'year' THEN
					DATEFROMPARTS(DATEPART(YEAR, al.AccessDateTime), 1, 1)
				WHEN 'ever' THEN
					NULL
			END AS TimeGroup
		INTO #LogData
		FROM
			NWIC_PanelMgmt.App.vNVCC_ProductionAccessLog al
			INNER JOIN CDWWork.Patient.Patient p
				ON al.PatientSID = p.PatientSID
			INNER JOIN Sta3n st
				ON (p.Sta3n = st.Sta3n OR st.Sta3n IS NULL)
			INNER JOIN UserID uid
				ON (al.UserID = uid.UserID OR uid.UserID IS NULL)

	-- A temporary table, #LogAgg, is created based on the structure of #LogAgg plus one
	-- other, NULLable numeric column N. This must be done in a separate step
	-- step (rather than implicitly with a SELECT ... INTO) because the same named
	-- temporary table can not be used as a target for multiple SELECT ... INTO statements.
	-- The WHERE clause is designed to select no rows, so no data is initially seeded
	-- in the temporary table.
	SELECT
		ld.*,
		CONVERT(integer, NULL) AS N
	INTO #LogAgg
	FROM #LogData AS ld
	WHERE 1=0;

	-- Choose the code path based on the value of @statistic.
	IF (@statistic = 'count')
		BEGIN
			INSERT INTO #LogAgg (Sta3n, UserID, TimeGroup, N)
			SELECT
				Sta3n,
				UserID,
				TimeGroup,
				COUNT(*) AS N
			FROM #LogData
			GROUP BY Sta3n, UserID, TimeGroup
		END

	IF (@statistic = 'cumulative')
		BEGIN
			INSERT INTO #LogAgg (Sta3n, UserID, TimeGroup, N)
			SELECT
				Sta3n,
				UserID,
				TimeGroup,
				SUM(N) OVER (PARTITION BY Sta3n, UserID ORDER BY TimeGroup) AS N
			FROM (
				SELECT
					Sta3n,
					UserID,
					TimeGroup,
					COUNT(*) AS N
				FROM #LogData
				GROUP BY Sta3n, UserID, TimeGroup
			) A
		END

	-- This temporary table is no longer needed.
	DROP TABLE #LogData;

	-- Further annotate the #LogAgg table by converting the UserID
	-- into the Staff Name and converting the station number into
	-- a more usable version of the station name based on the
	-- App.udf_StationName function.
	--
	-- An additional ID column is included so that Entity Framework
	-- has something to use as a primary key. Ironically, there is
	-- no column in the table which is guaranteed to be non-NULL
	-- for all values of the parameters. And apparently, having
	-- ID as the ORDER BY inside the OVER clause defining ID
	-- isn't invalid. I have no idea what it means, though.
	-- However, since the ID is not semantically meaningful (it
	-- just has to be unique), it doesn't matter.
	--
	-- Return this further annotated table.
	SELECT
		ROW_NUMBER() OVER (ORDER BY ID) AS ID,
		la.*,
		s.StaffName,
		App.udf_StationName(la.Sta3n) AS Station
	FROM
		#LogAgg AS la
		LEFT JOIN NWIC_PanelMgmt.App.NVCC_Users u
			ON la.UserID = u.DomainPlusNetworkUserName
		LEFT JOIN CDWWork.SStaff.SStaff s
			ON u.UserIEN = s.StaffIEN and u.Sta3n = s.Sta3n
	ORDER BY Sta3n, UserID, TimeGroup
END
GO

exec sp_SignAppObject 'NVCC_usp_UsageData2';

-- Examples/Tests:
--EXEC App.NVCC_usp_UsageData2 NULL, NULL, NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 NULL, 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 NULL, 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'day', 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'day', 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'day', 'count', '610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'day', 'cumulative', '610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'day', 'count', '648', 'DNS\DNS,VHA20\DNS ';
--EXEC App.NVCC_usp_UsageData2 'day', 'cumulative', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'week', 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'week', 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'week', 'count', '610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'week', 'cumulative', '610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'week', 'count', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'week', 'cumulative', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'month', 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'month', 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'month', 'count', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'month', 'cumulative', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'month', 'count', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'month', 'cumulative', '648', 'DNS\DNS,DNS\DNS   DNS';
--Exec App.NVCC_usp_UsageData2 'ever', 'count', null, null;
--Exec App.NVCC_usp_UsageData2 'ever', 'count', '648', null;
--Exec App.NVCC_usp_UsageData2 'ever', 'count', '648,610', null;
--EXEC App.NVCC_usp_UsageData2 'ever', 'count', '648', 'DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'ever', 'count', NULL, 'DNS\DNS   DNS';
--Exec App.NVCC_usp_UsageData2 'ever', 'cumulative', null, null;
--Exec App.NVCC_usp_UsageData2 'ever', 'cumulative', '648', null;
--Exec App.NVCC_usp_UsageData2 'ever', 'cumulative', '648,610', null;
--EXEC App.NVCC_usp_UsageData2 'ever', 'cumulative', '648', 'DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'ever', 'cumulative', NULL, 'DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'quarter', 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'quarter', 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'quarter', 'count', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'quarter', 'cumulative', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'quarter', 'count', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'quarter', 'cumulative', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'year', 'count', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'year', 'cumulative', NULL, NULL;
--EXEC App.NVCC_usp_UsageData2 'year', 'count', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'year', 'cumulative', '648,610,531', NULL;
--EXEC App.NVCC_usp_UsageData2 'year', 'count', '648', 'DNS\DNS,DNS\DNS   DNS';
--EXEC App.NVCC_usp_UsageData2 'year', 'cumulative', '648', 'DNS\DNS,DNS\DNS   DNS';
